package pl.model;

import javax.faces.validator.ValidatorException;
import javax.persistence.Column;
import javax.persistence.DiscriminatorColumn;
import javax.persistence.DiscriminatorType;
import javax.persistence.Entity;
import javax.persistence.EntityManager;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.Query;
import javax.persistence.Table;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

import pl.model.exception.UniquenessConstraintViolation;

@Entity
@Table( name = "persons")
@Inheritance( strategy = InheritanceType.JOINED)
@DiscriminatorColumn( name = "CATEGORY", discriminatorType = DiscriminatorType.STRING, length = 16)
public abstract class Person {
  @Id
  @NotNull( message = "A person ID value is required!")
  private Integer personId;
  @Column( nullable = false)
  @NotNull( message = "A name is required!")
  private String name;

  /**
   * Default constructor, required by @Entity annotation. This is needed since
   * other constructors are also defined so the default, no-parameters
   * constructors is not generated automatically at compile time by JRE.
   */
  public Person() {
  }

  /**
   * Create a new Person instance when the personId and name property values are
   * known.
   * 
   * @param personId
   *          the value to assign to the personId property
   * @param name
   *          the value to assign to the name property
   */
  public Person( Integer personId, String name) {
    this.setPersonId( personId);
    this.setName( name);
  }

  public Integer getPersonId() {
    return personId;
  }

  public void setPersonId( Integer personId) {
    this.personId = personId;
  }

  public String getName() {
    return name;
  }

  public void setName( String name) {
    this.name = name;
  }

  public String toString() {
    String result = "{ personId: " + this.personId + ", name:'" + this.name
        + "}";
    return result;
  }

  @Override
  public boolean equals( Object obj) {
    if ( obj instanceof Person) {
      Person person = (Person) obj;
      return (this.personId.equals( person.personId));
    } else
      return false;
  }

  /**
   * Check for the personId uniqueness constraint by verifying the existence in
   * the database of an person entry for the given personId value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param personId
   *          the personId of the person to check if exists or not
   * @throws ValidatorException
   * @throws UniquenessConstraintViolation
   */
  public static void checkPersonIdAsId( EntityManager em, Integer personId)
      throws UniquenessConstraintViolation {
    Person person = Person.retrieve( em, personId);
    // person was found, uniqueness constraint validation failed
    if ( person != null) {
      throw new UniquenessConstraintViolation(
          "There is already a person record with this personId!");
    }
  }

  /**
   * Read a Person entry from the <code>persons</code> database table.
   * 
   * @param em
   *          reference to the entity manager
   * 
   * @return the person with the given personId entry found in the database.
   */
  public static Person retrieve( EntityManager em, Integer personId) {
    Person person = em.find( Person.class, personId);
    if ( person != null) {
      System.out.println( "Person.retrieve: loaded person " + person);
    }
    return person;
  }

  /**
   * Delete a row from the <code>persons</code> table. The row is identified by
   * the value of the <code>personId</code> column.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the person row to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void destroy( EntityManager em, UserTransaction ut,
      Integer personId) throws NotSupportedException, SystemException,
      IllegalStateException, SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException {
    ut.begin();
    Person person = em.find( Person.class, personId);
    em.remove( person);
    ut.commit();
  }

  /**
   * Clear all entries from the <code>persons</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void clearData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Query deleteQuery = em.createQuery( "DELETE FROM Person");
    deleteQuery.executeUpdate();
    ut.commit();
  }
}