package pl.model;

import java.util.List;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.faces.model.SelectItem;
import javax.faces.validator.ValidatorException;
import javax.persistence.Column;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.Query;
import javax.persistence.Table;
import javax.persistence.TypedQuery;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

import pl.model.exception.UniquenessConstraintViolation;
import pl.model.validation.EmployeeValidator;

@Entity
@Table( name = "employees")
@DiscriminatorValue( value = "EMPLOYEE")
@ViewScoped
@ManagedBean( name = "employee")
@EmployeeValidator( message = "Employee of type Manager requires a department value!")
public class Employee extends Person {
  @Column( nullable = false)
  @NotNull( message = "An employee ID is required!")
  private Integer empNo;
  @Column( nullable = true, length = 32)
  @Enumerated( EnumType.STRING)
  private EmployeeTypeEL type;
  @Column( nullable = true, length = 64)
  private String department;

  /**
   * Default constructor, required by @Entity annotation. This is needed since
   * other constructors are also defined so the default, no-parameters
   * constructors is not generated automatically at compile time by JRE.
   */
  public Employee() {
  }

  /**
   * Create a new Employee.
   * 
   * @param personId
   *          the value to assign to the personId property
   * @param name
   *          the value to assign to the name property
   * @param empNo
   *          the value to assign to the empNo property
   * @param type
   *          the value to assign to the type property
   * @param department
   *          the value to assign to the department property
   */
  public Employee( Integer personId, String name, Integer empNo,
      EmployeeTypeEL type, String department) {
    super( personId, name);
    this.setEmpNo( empNo);
    this.setType( type);
    this.setDepartment( department);
  }

  public Integer getEmpNo() {
    return empNo;
  }

  public void setEmpNo( Integer empNo) {
    this.empNo = empNo;
  }

  public EmployeeTypeEL getType() {
    return type;
  }

  public void setType( EmployeeTypeEL type) {
    this.type = type;
  }

  public String getDepartment() {
    return department;
  }

  public void setDepartment( String department) {
    this.department = department;
  }

  public static Employee retrieve( EntityManager em, Integer personId) {
    Employee employee = em.find( Employee.class, personId);
    if ( employee != null) {
      System.out.println( "Employee.retrieve: loaded employee " + employee);
    }
    return employee;
  }

  /**
   * Create a human readable serialisation.
   * 
   * @return a string representing the employee serialisation.
   */
  public String toString() {
    String result = super.toString();
    result = result.substring( 0, result.length() - 2);
    return result + ", empNo: '" + this.empNo + ", type: '"
        + (this.type != null ? this.type.getLabel() : "") + "', department:'"
        + (this.department != null ? this.department : "") + "'}";
  }

  /**
   * Create and return the list of items containing the employee types which are
   * available for the employee.
   * 
   * @return the list of employee type items
   */
  public SelectItem[] getTypeItems() {
    SelectItem[] items = new SelectItem[EmployeeTypeEL.values().length];
    int i = 0;
    for ( EmployeeTypeEL et : EmployeeTypeEL.values()) {
      items[i++] = new SelectItem( et.name(), et.getLabel());
    }
    return items;
  }

  /**
   * Check for the empNo uniqueness constraint by verifying the existence in the
   * database of an employee entry for the given empNo value.
   * 
   * @param em
   *          entity manager reference
   * @param ut
   *          user transaction reference
   * @param empNo
   *          the empNo of the employee to check if exists or not
   * @throws ValidatorException
   * @throws UniquenessConstraintViolation
   */
  @SuppressWarnings( "unchecked")
  public static void checkEmpNoAsId( EntityManager em, Integer empNo)
      throws UniquenessConstraintViolation {
    Query q = em
        .createQuery( "SELECT e FROM Employee e WHERE e.empNo = :empNo");
    q.setParameter( "empNo", empNo);
    List<Employee> employees = q.getResultList();
    // employee was found, uniqueness constraint validation failed
    if ( employees.size() > 0) {
      throw new UniquenessConstraintViolation(
          "There is already an employee record with this employee number!");
    }
  }

  /**
   * Read all the Employee entries.
   * 
   * @param em
   *          reference to the entity manager
   * 
   * @return the list of all the Employee entries found in the database.
   */
  public static List<Employee> retrieveAll( EntityManager em) {
    TypedQuery<Employee> query = em.createQuery( "SELECT e FROM Employee e",
        Employee.class);
    List<Employee> employees = query.getResultList();
    System.out.println( "Employee.retrieveAll: " + employees.size()
        + " employees were loaded from DB.");
    return employees;
  }

  /**
   * Create and persist an Employee.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value
   * @param name
   *          the name value
   * @param empNo
   *          the employee number value
   * @param type
   *          the employee type value
   * @param department
   *          the department value for the employee
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut,
      Integer personId, String name, Integer empNo, EmployeeTypeEL type,
      String department) throws NotSupportedException, SystemException,
      IllegalStateException, SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException, EntityExistsException {
    ut.begin();
    Employee employee = new Employee( personId, name, empNo, type, department);
    em.persist( employee);
    ut.commit();
  }

  /**
   * Update an Employee, .
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the employee to update (this value it is not
   *          updated, but used to identify the employee/person to update)
   * @param name
   *          the name value of the employee to update
   * @param empNo
   *          the employee number value
   * @param type
   *          the employee type value
   * @param department
   *          the department value for the employee
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut,
      Integer personId, String name, Integer empNo, EmployeeTypeEL type,
      String department) throws NotSupportedException, SystemException,
      IllegalStateException, SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException {
    ut.begin();
    Employee employee = em.find( Employee.class, personId);
    if ( employee == null) {
      throw new EntityNotFoundException( "The employee with ID = " + personId
          + " was not found!");
    }
    employee.setName( name);
    employee.setEmpNo( empNo);
    employee.setType( type);
    employee.setDepartment( department);
    ut.commit();
  }

  /**
   * Delete an Employee when the person ID is known.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the employee to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void destroy( EntityManager em, UserTransaction ut,
      Integer personId) throws NotSupportedException, SystemException,
      IllegalStateException, SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException {
    ut.begin();
    Employee employee = em.find( Employee.class, personId);
    em.remove( employee);
    ut.commit();
  }
}
