package pl.model;

import java.util.List;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.Table;
import javax.persistence.TypedQuery;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

@Entity
@Table( name = "authors")
@DiscriminatorValue( value = "AUTHOR")
@ViewScoped
@ManagedBean( name = "author")
public class Author extends Person {
  @NotNull( message = "A biography is required!")
  private String biography;

  /**
   * Default constructor, required by @Entity annotation. This is needed since
   * other constructors are also defined so the default, no-parameters
   * constructors is not generated automatically at compile time by JRE.
   */
  public Author() {
  }

  /**
   * Create a new Author instance when the personId, name and biography values
   * are known.
   * 
   * @param personId
   *          the value to assign to the personId property
   * @param name
   *          the value to assign to the name property
   * @param biography
   *          the value to assign to the biography property
   */
  public Author( Integer personId, String name, String biography) {
    this.setPersonId( personId);
    this.setName( name);
    this.setBiography( biography);
  }

  public String getBiography() {
    return biography;
  }

  public void setBiography( String biography) {
    this.biography = biography;
  }

  public static Author retrieve( EntityManager em, Integer personId) {
    Author author = em.find( Author.class, personId);
    if ( author != null) {
      System.out.println( "Author.retrieve: loaded author " + author);
    }
    return author;
  }

  /**
   * Read all the Author entries from the <code>authors</code> database table.
   * 
   * @param em
   *          reference to the entity manager
   * 
   * @return the list of all the Author entries found in the database.
   */
  public static List<Author> retrieveAll( EntityManager em) {
    TypedQuery<Author> query = em.createQuery( "SELECT a FROM Author a",
        Author.class);
    List<Author> authors = query.getResultList();
    System.out.println( "Author.retrieveAll: " + authors.size()
        + " authors were loaded from DB.");
    return authors;
  }

  /**
   * Create a row in the <code>authors</code> table.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the author row to create
   * @param biography
   *          the biography value of the author row to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut,
      Integer personId, String name, String biography)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException, EntityExistsException {
    ut.begin();
    Author author = new Author( personId, name, biography);
    em.persist( author);
    ut.commit();
  }

  /**
   * Update a row in the <code>authors</code> table. The row is identified by
   * the value of the <code>personId</code> column. The <code>personId</code>
   * value is not changed, but only <code>name</code> and <code>biography</code>
   * .
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the author row to update
   * @param name
   *          the name value of the author row to update
   * @param biography
   *          the biography value of the author row to update
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut,
      Integer personId, String name, String biography)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Author author = em.find( Author.class, personId);
    if ( author == null) {
      throw new EntityNotFoundException( "The author with ID = " + personId
          + " was not found!");
    }
    author.setName( name);
    author.setBiography( biography);
    ut.commit();
  }

  /**
   * Delete a row from the <code>authors</code> table. The row is identified by
   * the value of the <code>personId</code> column.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the author row to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void destroy( EntityManager em, UserTransaction ut,
      Integer personId) throws NotSupportedException, SystemException,
      IllegalStateException, SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException {
    ut.begin();
    Author author = em.find( Author.class, personId);
    em.remove( author);
    ut.commit();
  }

}