package pl.ctrl;

import java.util.List;

import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.SessionScoped;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.validator.ValidatorException;
import javax.persistence.EntityExistsException;

import pl.model.Employee;
import pl.model.EmployeeTypeEL;
import pl.model.Person;
import pl.model.exception.UniquenessConstraintViolation;

@SessionScoped @ManagedBean( name="employeeCtrl")
public class EmployeeController extends PersonController {

  /**
   * Read the list of all employees from the database.
   * 
   * @return an instance of all the employee entries found in the database.
   */
  public List<Employee> getEmployees() {
    return Employee.retrieveAll( em);
  }

  /**
   * Update the reference object by setting its property values to match the one
   * existing in the database for the specific instance, identified by the
   * primary key value.
   **/
  public void refreshObject( Employee employee) {
    Employee foundEmployee = Employee.retrieve( em,
        employee.getPersonId());
    employee.setPersonId( foundEmployee.getPersonId());
    employee.setName( foundEmployee.getName());
    employee.setEmpNo( foundEmployee.getEmpNo());
    employee.setType( foundEmployee.getType());
    employee.setDepartment( foundEmployee.getDepartment());
  }

  /**
   * UI specific check for the empNo uniqueness constraint.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param value
   *          the value to be checked - in this case is the empNo
   * @throws ValidatorException
   */
  public void checkEmpNoAsId( FacesContext context, UIComponent component,
      Object value) throws ValidatorException {
    Integer empNo = (Integer) value;
    try {
      Employee.checkEmpNoAsId( em, empNo);
    } catch ( UniquenessConstraintViolation e) {
      throw new ValidatorException( new FacesMessage(
          FacesMessage.SEVERITY_ERROR, e.getMessage(), e.getMessage()));
    }
  }

  /**
   * Create and persist a new Employee instance.
   * 
   * @param personId
   *          the personId of the employee to create
   * @param name
   *          the name of the employee to create
   * @param empNo
   *          the employee number value
   * @param type
   *          the employee type value
   * @param department
   *          the department value for the employee
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String add( Integer personId, String name, Integer empNo,
      EmployeeTypeEL type, String department) {
    try {
      Employee.add( em, ut, personId, name, empNo, type, department);
      // Enforce clearing the form after creating the Employee.
      // Without this, the form will show the latest completed data
      FacesContext facesContext = FacesContext.getCurrentInstance();
      facesContext.getExternalContext().getRequestMap().remove( "employee");
    } catch ( EntityExistsException e) {
      try {
        ut.rollback();
      } catch ( Exception e1) {
        e1.printStackTrace();
      }
      e.printStackTrace();
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "create";
  }

  /**
   * Update an Employee instance.
   * 
   * @param personId
   *          the personId of the employee to update
   * @param name
   *          the name of the employee to update
   * @param empNo
   *          the employee number value
   * @param type
   *          the employee type value
   * @param department
   *          the department value for the employee
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String update( Integer personId, String name, Integer empNo,
      EmployeeTypeEL type, String department) {
    try {
      Employee.update( em, ut, personId, name, empNo, type, department);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "update";
  }

  /**
   * Delete an Employee entry from database.
   * 
   * @param personId
   *          the id of the employee to delete - used to uniquely identify the
   *          employee entry.
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String destroy( Integer personId) {
    try {
      Person.destroy( em, ut, personId);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "delete";
  }
}
