package pl.model;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.faces.validator.ValidatorException;
import javax.persistence.CacheStoreMode;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.Query;
import javax.persistence.Table;
import javax.persistence.TypedQuery;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

import pl.model.exception.UniquenessConstraintViolation;

@Entity @Table( name="publishers")
@ViewScoped @ManagedBean( name="publisher")
public class Publisher {
  @Id @NotNull( message="A name is required!")
  private String name;
  @Column( nullable=false)
  @NotNull( message="An address is required!")
  private String address;
  @OneToMany( fetch=FetchType.EAGER, mappedBy="publisher")
  private Set<Book> publishedBooks;

  /**
   * Default constructor, required for entity classes
   */
  public Publisher() {}

  /**
   * Constructor
   */
  public Publisher( String name, String address, Set<Book> publishedBooks) {
    this.setName( name);
    this.setAddress( address);
    this.setPublishedBooks( publishedBooks);
  }

  /**
   * Getters and setters
   */
  public String getName() {
    return name;
  }

  public void setName( String name) {
    this.name = name;
  }

  public String getAddress() {
    return address;
  }

  public void setAddress( String address) {
    this.address = address;
  }

  public Set<Book> getPublishedBooks() {
    return publishedBooks;
  }

  public void setPublishedBooks( Set<Book> publishedBooks) {
    Set<Book> newPubBooks = publishedBooks != null ? new HashSet<Book>(
        publishedBooks) : null;
    Set<Book> oldPubBooks = this.publishedBooks != null ? new HashSet<Book>(
        this.publishedBooks) : null;
    if ( oldPubBooks != null) {
      for ( Book b : oldPubBooks) {
        this.removePublishedBook( b);
      }
    }
    if ( newPubBooks != null) {
      for ( Book b : newPubBooks) {
        this.addPublishedBook( b);
      }
    } else {
      if ( this.publishedBooks != null) {
        this.publishedBooks.clear();
      }
    }
  }

  public void addPublishedBook( Book publishedBook) {
    if ( this.publishedBooks == null) {
      this.publishedBooks = new HashSet<Book>();
    }
    if ( !this.publishedBooks.contains( publishedBook)) {
      this.publishedBooks.add( publishedBook);
      publishedBook.setPublisher( this);
    }
  }

  public void removePublishedBook( Book publishedBook) {
    if ( this.publishedBooks != null && publishedBook != null
        && this.publishedBooks.contains( publishedBook)) {
      this.publishedBooks.remove( publishedBook);
      publishedBook.setPublisher( null);
    }
  }

  /**
   * Create a human readable serialization.
   */
  public String toString() {
    String result = "{ name: '" + this.name + "', address:'" + this.address
        + ", publishedBooks: [";
    if ( this.publishedBooks != null) {
      int i = 0, n = this.publishedBooks.size();
      for ( Book b : this.publishedBooks) {
        result += "'" + b.getTitle() + "'";
        if ( i < n - 1) {
          result += ", ";
        }
        i++;
      }
    }
    result += "]}";
    return result;
  }
  
  @Override
  public boolean equals( Object obj) {
    if (obj instanceof Publisher) {
      Publisher publisher = (Publisher) obj;
      return ( this.name.equals( publisher.name));
    } else return false;
  }

  /**
   * Check for the name uniqueness constraint by verifying the existence in the
   * database of a publisher entry for the given name value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param name
   *          the name of the publisher to check if exists or not
   * @throws ValidatorException
   * @throws UniquenessConstraintViolation
   */
  public static void checkNameAsId( EntityManager em, String name)
      throws UniquenessConstraintViolation {
    Publisher publisher = Publisher.retrieve( em, name);
    // publisher was found, uniqueness constraint validation failed
    if ( publisher != null) {
      throw new UniquenessConstraintViolation(
          "There is already a publisher record with this name!");
    }
  }

  /**
   * Retrieve all Publisher records from the publishers table.
   * 
   * @param em
   *          reference to the entity manager
   * @return all Publisher records
   */
  public static List<Publisher> retrieveAll( EntityManager em) {
    TypedQuery<Publisher> query = em.createQuery( "SELECT p FROM Publisher p", Publisher.class);
    List<Publisher> publishers = query.getResultList();
    System.out.println( "Publisher.retrieveAll: " + publishers.size()
        + " publishers were loaded from DB.");
    return publishers;
  }

  /**
   * Retrieve a publisher record from the publishers table.
   * 
   * @param em
   *          reference to the entity manager
   * @param name
   *          the publisher's name
   * @return the publisher with the given name
   */
  public static Publisher retrieve( EntityManager em, String name) {
    Publisher publisher = em.find( Publisher.class, name);
    if ( publisher != null) {
      System.out.println( "Publisher.retrieve: loaded publisher "
          + publisher);
    }
    return publisher;
  }

  /**
   * Helper method used with Publisher.add and Publisher.destroy to update the
   * references of the old publishers of a book which was selected as being
   * published by a new author (this is required since a Book can have none or
   * just one publisher)
   * 
   * @param em
   *          reference to the entity manager for query reasons
   * @param books
   *          the books to remove from old publishers
   */
  @SuppressWarnings( "unchecked")
  private static void updatePublishersForOldBooks( EntityManager em,
      Set<Book> books) {
    List<String> bookIsbns = new ArrayList<String>();
    for ( Book b : books) {
      bookIsbns.add( b.getIsbn());
    }
    Query query = em.createQuery(
        "SELECT b FROM Book b WHERE b.isbn IN :bookIsbns", Book.class);
    // enforce to load the books directly from DB - non-cached versions
    query.setHint( "javax.persistence.cache.storeMode", CacheStoreMode.REFRESH);
    // add titles of the books to load from DB
    query.setParameter( "bookIsbns", bookIsbns);
    List<Book> nonCachedBooks = (List<Book>) query.getResultList();
    for ( Book b : nonCachedBooks) {
      Publisher p = b.getPublisher();
      if ( p != null) {
        p.removePublishedBook( b);
        em.merge( p);
      }
    }

  }

  /**
   * Create a Publisher instance.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param name
   *          the name value of the publisher to create
   * @param address
   *          the address value of the publisher to create
   * @param publishedBooks
   *          the publishedBooks value of the publisher to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut, String name,
      String address, Set<Book> publishedBooks) throws NotSupportedException,
      SystemException, IllegalStateException, SecurityException,
      HeuristicMixedException, HeuristicRollbackException, RollbackException,
      EntityExistsException {
    ut.begin();
    // get old publishers of the published books to merge their references
    Publisher.updatePublishersForOldBooks( em, publishedBooks);
    // create and persist the new publisher
    Publisher publisher = new Publisher( name, address, publishedBooks);
    em.persist( publisher);
    // update referenced books
    if ( publisher.getPublishedBooks() != null) {
      for ( Book b : publisher.getPublishedBooks()) {
        em.merge( b);
      }
    }
    ut.commit();
    System.out.println( "Publisher.add: the publisher " + publisher
        + " was saved.");
  }

  /**
   * Update a Publisher instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param name
   *          the name of the publisher to update
   * @param address
   *          the address of the publisher to update
   * @param publishedBooks
   *          the published books of the publisher to update
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut, String name,
      String address, Set<Book> publishedBooks) throws NotSupportedException,
      SystemException, IllegalStateException, SecurityException,
      HeuristicMixedException, HeuristicRollbackException, RollbackException {

    ut.begin();
    // get old publishers of the published books to merge their references
    Publisher.updatePublishersForOldBooks( em, publishedBooks);
    // obtain reference to the publisher to update
    Publisher publisher = em.find( Publisher.class, name);
    if ( publisher == null) {
      throw new EntityNotFoundException( "The publisher with name = " + name
          + " was not found!");
    }
    if ( address != null && !address.equals( publisher.address)) {
      publisher.setAddress( address);
    }
    if ( publishedBooks != null
        && !publishedBooks.equals( publisher.publishedBooks)) {
      publisher.setPublishedBooks( publishedBooks);
    }
    // update cached book references
    for ( Book b : publisher.getPublishedBooks()) {
      em.merge( b);
    }
    ut.commit();
    System.out.println( "Publisher.update: the publisher " + publisher
        + " was updated.");
  }

  /**
   * Delete a Publisher instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param name
   *          the name value of the publisher to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void destroy( EntityManager em, UserTransaction ut, String name)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Publisher publisher = em.find( Publisher.class, name);
    publisher.setPublishedBooks( null);
    em.remove( publisher);
    ut.commit();
    System.out.println( "Publisher.destroy: the publisher " + publisher
        + " was deleted.");
  }

  /**
   * Clear all entries from the <code>publishers</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void clearData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Query deleteQuery = em.createQuery( "DELETE FROM Publisher");
    deleteQuery.executeUpdate();
    ut.commit();
  }
}