package pl.model;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.faces.validator.ValidatorException;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.ManyToMany;
import javax.persistence.Query;
import javax.persistence.Table;
import javax.persistence.TypedQuery;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

import pl.model.exception.UniquenessConstraintViolation;

@Entity @Table( name="authors")
@ViewScoped @ManagedBean( name="author")
public class Author {
  @Id private Integer personId;
  @Column( nullable=false)
  @NotNull( message="A name is required!")
  private String name;
  @ManyToMany( fetch=FetchType.EAGER, mappedBy="authors")
  private Set<Book> authoredBooks;

  /**
   * Default constructor, required for entity classes
   */
  public Author() {}

  /**
   * Constructor
   */
  public Author( Integer personId, String name, Set<Book> authoredBooks) {
    this.setPersonId( personId);
    this.setName( name);
    this.setAuthoredBooks( authoredBooks);
  }

  /**
   * Getters and setters
   */
  public Integer getPersonId() {
    return personId;
  }

  public void setPersonId( Integer personId) {
    this.personId = personId;
  }

  public String getName() {
    return name;
  }

  public void setName( String name) {
    this.name = name;
  }

  public Set<Book> getAuthoredBooks() {
    return authoredBooks;
  }

  public void setAuthoredBooks( Set<Book> authoredBooks) {
    Set<Book> oldAuthBooks = this.authoredBooks != null ? new HashSet<Book>(
        this.authoredBooks) : null;
    Set<Book> newAuthBooks = authoredBooks != null ? new HashSet<Book>(
        authoredBooks) : null;
    if ( oldAuthBooks != null) {
      for ( Book b : oldAuthBooks) {
        this.removeAuthoredBook( b);
      }
    }
    // add this book to the authoredBooks of the new authors of this book
    if ( newAuthBooks != null) {
      for ( Book b : newAuthBooks) {
        this.addAuthoredBook( b);
      }
    } else {
      if ( this.authoredBooks != null) {
        this.authoredBooks.clear();
      }
    }
  }

  public void addAuthoredBook( Book authoredBook) {
    if ( this.authoredBooks == null) {
      this.authoredBooks = new HashSet<Book>();
    }
    if ( !this.authoredBooks.contains( authoredBook)) {
      this.authoredBooks.add( authoredBook);
      authoredBook.addAuthor( this);
    }
  }

  public void removeAuthoredBook( Book authoredBook) {
    if ( this.authoredBooks != null && authoredBook != null
        && this.authoredBooks.contains( authoredBook)) {
      this.authoredBooks.remove( authoredBook);
      authoredBook.removeAuthor( this);
    }
  }

  /**
   * Create a human readable serialization.
   */
  public String toString() {
    String result = "{ personId: " + this.personId + ", name:'" + this.name;
    result += ", authoredBooks: [";
    if ( this.authoredBooks != null) {
      int i = 0, n = this.authoredBooks.size();
      for ( Book b : this.authoredBooks) {
        result += "'" + b.getTitle() + "'";
        if ( i < n - 1) {
          result += ", ";
        }
        i++;
      }
    }
    result += "]}";
    return result;
  }
  
  @Override
  public boolean equals( Object obj) {
    if (obj instanceof Author) {
      Author author = (Author) obj;
      return ( this.personId.equals( author.personId));
    } else return false;
  }

  /**
   * Check for the personId uniqueness constraint by verifying the existence in
   * the database of an author entry for the given personId value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param personId
   *          the personId of the author to check if exists or not
   * @throws ValidatorException
   * @throws UniquenessConstraintViolation
   */
  public static void checkPersonIdAsId( EntityManager em, Integer personId)
      throws UniquenessConstraintViolation {
    Author author = Author.retrieve( em, personId);
    // author was found, uniqueness constraint validation failed
    if ( author != null) {
      throw new UniquenessConstraintViolation(
          "There is already an author record with this personId!");
    }
  }

  /**
   * Retrieve all Author records from the authors table.
   * 
   * @param em
   *          reference to the entity manager
   * @return all Author records
   */
  public static List<Author> retrieveAll( EntityManager em) {
    TypedQuery<Author> query = em.createQuery( "SELECT a FROM Author a", Author.class);
    List<Author> authors = query.getResultList();
    System.out.println( "Author.retrieveAll: " + authors.size()
        + " authors were loaded from DB.");
    return authors;
  }

  /**
   * Retrieve an Author record from the authors table.
   * 
   * @param em
   *          reference to the entity manager
   * @param personId
   *          the author's personId ISBN
   * @return the Author with the given personId
   */
  public static Author retrieve( EntityManager em, Integer personId) {
    Author author = em.find( Author.class, personId);
    if ( author != null) {
      System.out.println( "Author.retrieve: loaded author " + author);
    }
    return author;
  }

  /**
   * Create an Author instance.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the author to create
   * @param name
   *          the name value of the author to create
   * @param authoredBooks
   *          the authoredBooks value of the author to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut,
      Integer personId, String name, Set<Book> authoredBooks) 
          throws NotSupportedException, SystemException, IllegalStateException, 
          SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException, EntityExistsException {
    ut.begin();
    Author author = new Author( personId, name, authoredBooks);
    em.persist( author);
    for ( Book b : author.getAuthoredBooks()) {
      em.merge( b);
    }
    ut.commit();
    System.out.println( "Author.add: the author " + author + " was saved.");
  }

  /**
   * Update an Author instance
   * 
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the author to update
   * @param name
   *          the name value of the author to update
   * @param authoredbooks
   *          the authoredBooks value of the author to update
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut,
      Integer personId, String name, Set<Book> authoredBooks) 
          throws NotSupportedException, SystemException, IllegalStateException, 
          SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException {
    ut.begin();
    Author author = em.find( Author.class, personId);
    if ( author == null) {
      throw new EntityNotFoundException( "The author with ID = " + personId
          + " was not found!");
    }
    // delete the references of the updated author from all the books authored
    // by this author before the current update operation
    for ( Book b : author.getAuthoredBooks()) {
      b.removeAuthor( author);
      em.merge( b);
    }
    if ( name != null && !name.equals( author.name)) {
      author.setName( name);
    }
    if ( authoredBooks != null && !authoredBooks.equals( author.authoredBooks)) {
      author.setAuthoredBooks( authoredBooks);
    }
    // update the ManyToMany reference: the authors for the book
    for ( Book b : authoredBooks) {
      b.addAuthor( author);
      em.merge( b);
    }
    ut.commit();
    System.out.println( "Author.update: the author " + author + " was updated.");
  }

  /**
   * Delete an Author instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param personId
   *          the personId value of the author to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void destroy( EntityManager em, UserTransaction ut,
      Integer personId) throws NotSupportedException, SystemException,
      IllegalStateException, SecurityException, HeuristicMixedException,
      HeuristicRollbackException, RollbackException {
    ut.begin();
    Author author = em.find( Author.class, personId);
    author.setAuthoredBooks( null);
    em.remove( author);
    ut.commit();
    System.out.println( "Author.destroy: the author " + author + " was deleted.");
  }

  /**
   * Clear all entries from the <code>authors</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void clearData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Query deleteQuery = em.createQuery( "DELETE FROM Author");
    deleteQuery.executeUpdate();
    ut.commit();
  }
}